package com.example.aidemo;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.openai.client.OpenAIClient;
import com.openai.models.chat.completions.ChatCompletion;
import com.openai.models.chat.completions.ChatCompletionAssistantMessageParam;
import com.openai.models.chat.completions.ChatCompletionCreateParams;
import com.openai.models.chat.completions.ChatCompletionMessageParam;
import com.openai.models.chat.completions.ChatCompletionSystemMessageParam;
import com.openai.models.chat.completions.ChatCompletionUserMessageParam;
import jakarta.servlet.http.HttpSession;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

@Controller
public class ChatController {

    @Autowired
    private OpenAIClient client;

    @GetMapping("/")
    public String index() {
        return "chat";
    }

    @PostMapping("/chat/send")
    @ResponseBody
    public Map<String, Object> sendMessage(@RequestParam String message, HttpSession session) {
        // 1. 获取会话历史 (存储 ChatCompletionMessageParam 以便直接传给 SDK)
        List<ChatCompletionMessageParam> history = (List<ChatCompletionMessageParam>) session.getAttribute("history");
        if (history == null) {
            history = new ArrayList<>();
            // 可选：添加系统预设
            history.add(ChatCompletionMessageParam.ofSystem(
                    ChatCompletionSystemMessageParam.builder().content("你是一个专业的编程助手").build()
            ));
        }

        // 2. 添加用户消息
        history.add(ChatCompletionMessageParam.ofUser(
            ChatCompletionUserMessageParam.builder().content(message).build()
        ));

        // 3. 滑动窗口：保留最近 10 条对话记录
        if (history.size() > 11) { // 1条系统消息 + 10条对话
            history = new ArrayList<>(history.subList(history.size() - 11, history.size()));
            history.set(0, ChatCompletionMessageParam.ofSystem(
                    ChatCompletionSystemMessageParam.builder().content("你是一个专业的编程助手").build()
            ));
        }

        try {
            // 4. 调用 AI
            ChatCompletionCreateParams params = ChatCompletionCreateParams.builder()
                    .messages(history)
                    .model("qwen-plus")
                    .build();

            ChatCompletion completion = client.chat().completions().create(params);
            String aiResponse = completion.choices().get(0).message().content().orElse("");

            // 5. 保存 AI 回复并更新 Session
            history.add(ChatCompletionMessageParam.ofAssistant(
                ChatCompletionAssistantMessageParam.builder().content(aiResponse).build()
            ));
            session.setAttribute("history", history);

            return Map.of("success", true, "data", aiResponse);
        } catch (Exception e) {
            return Map.of("success", false, "message", e.getMessage());
        }
    }
}
